import os
import uuid
from datetime import datetime, timedelta
from functools import wraps
from os.path import isfile, join

import jwt
from flask import Flask, make_response, redirect, render_template, request
from flask_migrate import Migrate
from flask_sqlalchemy import SQLAlchemy
from werkzeug.security import check_password_hash, generate_password_hash

app = Flask(__name__)

app.config["SECRET_KEY"] = os.environ.get("SECRET_KEY")

app.config["SQLALCHEMY_DATABASE_URI"] = os.environ.get("SQLALCHEMY_DATABASE_URI")
app.config["SQLALCHEMY_TRACK_MODIFICATIONS"] = True

db = SQLAlchemy(app)
migrate = Migrate(app, db)


@app.after_request
def add_security_headers(resp):
    resp.headers["Content-Security-Policy"] = "default-src 'self'"
    return resp


class User(db.Model):
    id = db.Column(db.Integer, primary_key=True)
    name = db.Column(db.String(1000))
    password = db.Column(db.String(800))
    notes = db.relationship("Note", backref="user", lazy=True)


class Note(db.Model):
    id = db.Column(db.String(1000), primary_key=True)
    content = db.Column(db.Text)
    user_id = db.Column(db.Integer, db.ForeignKey("user.id"), nullable=False)


def token_required(f):
    @wraps(f)
    def decorated(*args, **kwargs):
        token = request.cookies.get("session")

        if not token:
            return redirect("/login")

        try:
            data = jwt.decode(token, app.config["SECRET_KEY"])
            current_user = User.query.filter_by(name=data["name"]).first()
        except:
            return redirect("/login")
        return f(current_user, *args, **kwargs)

    return decorated


@app.route("/")
@token_required
def index(current_user):
    return render_template("/index.html", user=current_user, flag=flag)


@app.route("/add_note", methods=["POST"])
@token_required
def add_note(current_user):
    content = request.form.get("content")
    id = str(uuid.uuid4())
    note = Note(id=id, content=content, user_id=current_user.id)
    db.session.add(note)
    db.session.commit()
    return redirect(f"/note?id={id}")


@app.route("/note/")
@token_required
def note(current_user):
    id = request.args.get("id")
    note = Note.query.filter_by(id=id).first()
    if note is None:
        return render_template("404.html", id=id)
    return render_template("note.html", note=note)


@app.route("/login", methods=["GET", "POST"])
def login():
    if request.method == "GET":
        return render_template("login.html")
    auth = request.form

    if not auth or not auth.get("name") or not auth.get("password"):
        return redirect("/login")

    user = User.query.filter_by(name=auth.get("name")).first()

    if not user:
        return make_response(
            "Wrong username or password",
            401,
        )

    if check_password_hash(user.password, auth.get("password")):
        token = jwt.encode(
            {"name": user.name, "exp": datetime.utcnow() + timedelta(minutes=30)},
            app.config["SECRET_KEY"],
        )

        resp = redirect("/")
        resp.set_cookie("session", token.decode("UTF-8"))
        return resp

    return make_response(
        "Wrong username or password",
        401,
    )


# signup route
@app.route("/signup", methods=["GET", "POST"])
def signup():
    if request.method == "GET":
        return render_template("signup.html")

    data = request.form

    name = data.get("name")
    password = data.get("password")
    if not name or not password:
        return redirect("/signup")

    user = User.query.filter_by(name=name).first()
    if not user:
        user = User(name=name, password=generate_password_hash(password))
        db.session.add(user)
        db.session.commit()

        return make_response("Successfully registered.")
    else:
        return make_response("User already exists. Please Log in.")

if __name__ == "__main__":
    app.run("0.0.0.0")
